from datetime import datetime, timedelta
from .loaders import FileNotFound, LoaderError
from .datetime import local_to_aware

class FileCache:
  def __init__(self):
    self.cache =  {}

  def get(self, name):
    return self.cache.get(name, (None, None, None))

  def set(self, name, contents, last_check, mtime):
    self.cache[name] = (contents, last_check, mtime)

class CachedFileReader:
  cache = FileCache()

  def __init__(self, remote_loader, local_loader, reload_after_minutes):
    self.remote_loader = remote_loader
    self.local_loader  = local_loader
    self.reload_after_minutes = reload_after_minutes

  def __call__(self, name, mode = 'rb'):
    now = local_to_aware(datetime.now())

    # load from cache
    (cached, last_check, stored_mtime) = self.cache.get(name)

    if cached:
      if (now - last_check) < timedelta(minutes = self.reload_after_minutes):
        # cached content is recent enough

        if isinstance(cached, Exception):
          raise cached # cached 404
        else:
          return cached

      else: # cached content is outdated
        try:
          # load from server
          (contents, mtime) = self.remote_loader.read_if_modified_after(name, stored_mtime, mode)

          if contents: # content has changed
            # cache on disk
            self.local_loader.write(name, contents, mode.replace('r', 'w'))

            # cache in memory
            self.cache.set(name, contents, now, mtime)

            return contents

          else: # content hasn't changed
            # update cached last_check timestamp
            self.cache.set(name, cached, now, stored_mtime)

            return cached

        except LoaderError:
          # can't load - use cached anyway
          if isinstance(cached, Exception):
            raise cached # cached 404
          else:
            return cached

    else:
      try:
        # load from server
        (contents, mtime) = self.remote_loader.read_with_mtime(name, mode)

        # cache on disk
        self.local_loader.write(name, contents, mode.replace('r', 'w'))

        # cache in memory
        self.cache.set(name, contents, now, mtime)

        return contents

      except FileNotFound as exception:
        # cache 404 in memory
        self.cache.set(name, exception, now, now - timedelta(days = 365))

        # delete cached file from disk
        self.local_loader.delete(name)

        raise exception

      except LoaderError as remote_exception:
        try:
          # load from disk
          (contents, mtime) = self.local_loader.read_with_mtime(name, mode)

          # cache in memory
          self.cache.set(name, contents, now, mtime)

          return contents

        except FileNotFound as local_exception:
          # if local loading fails, raise original exception from remote loading attempt, so that we
          # can debug it
          raise remote_exception
