from django.db.models import Prefetch
from django.forms import Form, DateField, ModelMultipleChoiceField
from rest_framework.views import APIView
from rest_framework.response import Response
from alto_django_utils.permissions import AllowAny, IsClientUser
from alto_django_canteen.models import Canteen, Meal
from alto_django_canteen.serializers import MealSerializer

class LoadMenuForm(Form):
  canteens  = ModelMultipleChoiceField(required = False, queryset = Canteen.objects.all())
  from_date = DateField()
  to_date   = DateField()

class LoadMenu(APIView):
  permission_classes = [AllowAny]

  def get(self, request):
    form = LoadMenuForm(request.query_params)

    if form.is_valid():
      canteens  = form.cleaned_data['canteens']  or Canteen.objects.all()
      from_date = form.cleaned_data['from_date']
      to_date   = form.cleaned_data['to_date']

      return Response(self.serialize_meals(self.get_meals(canteens, from_date, to_date)))
    else:
      return Response(form.errors, status = 400)

  @staticmethod
  def get_meals(canteens, from_date, to_date):
    return Meal.objects \
      .filter(show_on_web = True, canteen__in = canteens,
              date__gte = from_date, date__lte = to_date) \
      .order_by('canteen_id', 'course__group__order_number',
                'course__order_number', 'number')

  def serialize_meals(self, meals):
    return MealSerializer(meals, many = True, envelope = True,
                          request_fields = self.fields_to_sideload()).data

  @staticmethod
  def fields_to_sideload():
    return {'canteen': {}, 'course': {'group': {}}}

class LoadMenuWithOrders(LoadMenu):
  permission_classes = [IsClientUser]

  def __init__(self):
    super().__init__()

    self.__request = None

  def get(self, request):
    self.__request = request

    return super().get(request)

  def get_meals(self, *args, **kwargs):
    # only sideload orders belonging to logged-in client
    return super().get_meals(*args, **kwargs) \
      .prefetch_related(Prefetch('orders', queryset = self.__request.user.client.orders.all()))

  @staticmethod
  def fields_to_sideload():
    return {'canteen': {}, 'course': {'group': {}}, 'orders': {}}
