import pydash
from django.db import transaction
from django.utils.translation import gettext as _
from rest_framework.views import APIView
from rest_framework.status import HTTP_400_BAD_REQUEST
from rest_framework.response import Response
from rest_framework.decorators import api_view, permission_classes
from alto_django_utils.permissions import IsClientUser
from alto_django_canteen.models import Order
from alto_django_canteen.serializers import (OrderSerializer, OrderCourseGroupSerializer,
                                             CancelCourseGroupSerializer)
from alto_django_canteen.services.course_group_ordering import OrderCourseGroupService as Service

messages = {
  Service.OrdersInOtherCanteenExist:         'Orders in other canteen exist',
  Service.OrderIsAlreadyBeingCreated:        'Order is already being created',
  Service.MainMealNotOrdered:                'Main meal not ordered',
  Service.AdditionalMealNotOrdered:          'Additional meal not ordered',
  Service.DefaultAdditionalMealDoesNotExist: "Default additional meal doesn't exist"
}

def to_bad_request_data(error):
  code    = pydash.snake_case(error.__class__.__name__)
  message = _(messages[error.__class__])

  if isinstance(error, Service.OrdersInOtherCanteenExist):
    return { 'canteen': [message], 'code': code }
  else:
    return { 'orders': [{'course': error.meal.course.name, 'detail': message, 'code': code}] }

@api_view(['POST'])
@permission_classes([IsClientUser])
def order_course_group(request):
  serializer = OrderCourseGroupSerializer(data = request.data)

  if serializer.is_valid():
    service = Service(request.user.client, **serializer.validated_data)

    try:
      return Response({ 'orders': OrderSerializer(service(), many = True).data })
    except Service.Error as error:
      return Response(to_bad_request_data(error), status = HTTP_400_BAD_REQUEST)
  else:
    return Response(dict(serializer.errors), status = HTTP_400_BAD_REQUEST)

@api_view(['DELETE'])
@permission_classes([IsClientUser])
def cancel_course_group(request):
  serializer = CancelCourseGroupSerializer(data = request.data)

  if serializer.is_valid():
    date, canteen, group = pydash.at(serializer.validated_data, 'date', 'canteen', 'group')

    orders = Order.objects.filter(client = request.user.client, meal__date = date,
                                  meal__canteen = canteen, meal__course__group = group)

    orders.delete()

    return Response({ 'orders': OrderSerializer(orders, many = True).data })
  else:
    return Response(dict(serializer.errors), status = 400)

class CourseGroupOrderingFailed(Exception):
  def __init__(self, course_group_index, service_error):
    self.course_group_index = course_group_index
    self.service_error      = service_error

class BulkOrder(APIView):
  permission_classes = [IsClientUser]

  def post(self, request):
    serializer = OrderCourseGroupSerializer(data = request.data, many = True)

    if serializer.is_valid():
      resulting_orders = []

      try:
        orders = self._preprocess_orders(serializer.validated_data)

        with transaction.atomic():
          for index, group_data in enumerate(orders):
            service = Service(request.user.client, **group_data)

            try:
              resulting_orders.extend(service())
            except Service.Error as error:
              raise CourseGroupOrderingFailed(index, error) # roll back transaction

      except CourseGroupOrderingFailed as error:
        data                           = [{}] * len(serializer.validated_data)
        data[error.course_group_index] = to_bad_request_data(error.service_error)

        return Response(data, status = HTTP_400_BAD_REQUEST)

      return Response({ 'orders': OrderSerializer(resulting_orders, many = True).data })
    else:
      return Response(serializer.errors, status = 400)

  def _preprocess_orders(self, orders):
    def has_cancellation(group_order):
      return not all(order['meal'] for order in group_order['orders'])

    grouped = pydash.group_by(orders, has_cancellation)

    return grouped.get(True, []) + grouped.get(False, [])
