from datetime                            import date, timedelta
from django.urls                         import reverse
from django.utils.functional             import cached_property
from alto_django_base_auth.tests.helpers import BaseAuthHelpers

class CanteenHelpers:
  use_jwt = False

  def __init__(self, auth_helpers = None, client = None, use_jwt = True):
    self.__auth_helpers = auth_helpers
    self.client         = client
    self.use_jwt        = use_jwt

  @cached_property
  def _auth_helpers(self):
    return getattr(self, '__auth_helpers', BaseAuthHelpers(self.client, self.use_jwt))

  def load_menu(self, *, from_date = None, to_date = None, with_orders = False, token = None):
    headers = self._auth_helpers.auth_header(token) if token else {}
    params  = {}

    if not from_date: from_date = date.today()
    if not to_date:   to_date   = from_date + timedelta(days = 7)

    params['from_date'] = from_date.isoformat()
    params['to_date']   = to_date.isoformat()

    view_name = 'load-menu-with-orders' if with_orders else 'load-menu'

    return self.client.get(reverse('alto_django_canteen:' + view_name), params, **headers)

  def order_meal(self, date, canteen, course, number, token):
    return self.client.post(reverse('alto_django_canteen:order-meal'),
                            {'date':    date.isoformat(),
                             'canteen': canteen.id,
                             'course':  course.name,
                             'number':  number},
                            **self._auth_helpers.auth_header(token))

  def cancel_meal(self, date, canteen, course, token):
    return self.client.delete(reverse('alto_django_canteen:cancel-meal'),
                              {'date':    date.isoformat(),
                               'canteen': canteen.id,
                               'course':  course.name},
                              **self._auth_helpers.auth_header(token))

  def preorder_meals(self, *, client, canteen, from_date, to_date, from_course, to_course,
                     only_courses = None, except_courses = None, token):
    data = {'client':      client.id,
            'canteen':     canteen.id,
            'from_date':   from_date,
            'from_course': from_course,
            'to_date':     to_date,
            'to_course':   to_course}

    if only_courses:   data['only_courses']   = only_courses
    if except_courses: data['except_courses'] = except_courses

    return self.client.post(reverse('alto_django_canteen:preorder-meals'), data,
                            **self._auth_helpers.auth_header(token))

  def order_course_group(self, date, canteen, group, orders, token):
    return self.client.post(reverse('alto_django_canteen:order-course-group'),
                            {'date':    date.isoformat(),
                             'canteen': canteen.id,
                             'group':   group.name,
                             'orders':  orders},
                            **self._auth_helpers.auth_header(token))

  def cancel_course_group(self, date, canteen, group, token):
    return self.client.delete(reverse('alto_django_canteen:cancel-course-group'),
                              {'date':    date.isoformat(),
                               'canteen': canteen.id,
                               'group':   group.name},
                              **self._auth_helpers.auth_header(token))

  def bulk_order(self, orders, token):
    return self.client.post(reverse('alto_django_canteen:bulk-order'),
                            orders, **self._auth_helpers.auth_header(token))

  def register_terminal(self, *args, **kwargs):
    return self._create_or_update_terminal('create', *args, **kwargs)

  def update_terminal(self, id, *args, **kwargs):
    return self._create_or_update_terminal('update', id, *args, **kwargs)

  def _create_or_update_terminal(self, action, id = None, *,
                                 name = None, ip = None, secret = None, token):
    attributes = {}

    if name:   attributes['name']   = name
    if ip:     attributes['ip']     = ip
    if secret: attributes['secret'] = secret

    method    = {'create': 'post', 'update': 'put'}[action]
    view_type = {'create': 'list', 'update': 'detail'}[action]
    args      = {'create': [],     'update': [id]}[action]

    return getattr(self.client, method)(
      reverse(f'alto_django_canteen:terminals-{view_type}', args = args),
      attributes, **self._auth_helpers.auth_header(token))

  def is_registered_terminal(self, secret = None):
    return self.client.get(reverse('alto_django_canteen:is-registered-terminal'), { 'secret': secret })

  def get_terminal_token(self, card_number, secret = None):
    url_name = 'get-terminal-token-pair' if self._auth_helpers.use_jwt else 'get-terminal-token'

    return self.client.post(reverse(f'alto_django_canteen:{url_name}'),
                            { 'card_number': card_number, 'secret': secret })
