import pytest
from datetime import date
from threading import Thread
from django.db import connection
from django.contrib.auth import get_user_model
from rest_framework.test import APITransactionTestCase
from alto_django_client_auth.tests.helpers import ClientAuthHelpers
from alto_django_kredit.models import Client
from alto_django_canteen.models import Canteen, CourseGroup, Course, Meal, Order
from alto_django_canteen.tests.helpers import CanteenHelpers
from alto_django_canteen.tests.factories import MealFactory, CourseFactory

User = get_user_model()

# NOTES:
# APITransactionTestCase doesn't restore initial db data after test, only at the end of test suite
# run -> tests that rely on data created in migrations

@pytest.mark.nodbreset
class OrderingConcurrencyTests(APITransactionTestCase, ClientAuthHelpers, CanteenHelpers):
  def setUp(self):
    self.client_    = Client.objects.create(client_id = '123', card_id = '123', email = 'client@test.cz')
    self.user       = User.objects.create_user('test', 'test@test.cz', 'test', client = self.client_)
    self.user_token = self.log_in(username = 'test',  password = 'test' ).data['token']

  def test(self):
    canteen = Canteen.objects.create(name = 'canteen')
    today   = date.today()
    course  = Course.objects.get(name = 'breakfast')
    meal    = MealFactory.create(canteen = canteen, date = today, course = course, number = 1)

    orders = [
      {'canteen': canteen.id,
       'date':    today.isoformat(),
       'group':   'breakfast',
       'orders':  [{'course': 'breakfast', 'number': 1}]},
    ]

    threads = [Thread(target = self._make_order, args = [orders]) for i in range(10)]

    for thread in threads: thread.start()
    for thread in threads: thread.join()

    assert Order.objects.count() == 1

  def _make_order(self, orders):
    response = self.bulk_order(orders, self.user_token)
    connection.close() # django creates new db connection for each thread, stop it from dangling
