from faker import Faker
from datetime import date, timedelta
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_utils.counter import Counter
from alto_django_client_auth.tests.helpers import ClientAuthHelpers
from alto_django_kredit.models import Client
from alto_django_canteen.models import Canteen, Meal, Course, Order
from alto_django_canteen.tests.helpers import CanteenHelpers

fake = Faker()
User = get_user_model()

one_day  = timedelta(days = 1)
two_days = timedelta(days = 2)

class MealPreorderingTests(APITestCase, ClientAuthHelpers, CanteenHelpers):
  def setUp(self):
    self.client_     = Client.objects.create(client_id = '123', card_id = '123',
                                             email = 'client@test.cz')
    self.user        = User.objects.create_superuser('admin', 'admin@test.cz', 'admin')
    self.admin_token = self.get_token(username = 'admin', password = 'admin' ).data['token']

    self.canteen     = Canteen.objects.create(name = 'canteen')
    self.breakfast   = Course.objects.get(name = 'breakfast')
    self.lunch       = Course.objects.get(name = 'lunch')
    self.dinner      = Course.objects.get(name = 'dinner')

  def test_one_day(self):
    self.create_meals(number_of_days = 1)

    response = self.preorder_meals(client      = self.client_,
                                   canteen     = self.canteen,
                                   from_date   = date.today(),
                                   to_date     = date.today(),
                                   from_course = self.lunch.name,
                                   to_course   = self.lunch.name,
                                   token       = self.admin_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 1

    assert not self.order_exists(self.client_, date.today(), self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today(), self.lunch,     1)
    assert not self.order_exists(self.client_, date.today(), self.dinner,    1)

  def test_two_days(self):
    self.create_meals(number_of_days = 2)

    response = self.preorder_meals(client      = self.client_,
                                   canteen     = self.canteen,
                                   from_date   = date.today(),
                                   to_date     = date.today() + one_day,
                                   from_course = self.lunch.name,
                                   to_course   = self.lunch.name,
                                   token       = self.admin_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 2 + 2

    assert not self.order_exists(self.client_, date.today(),           self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today(),           self.lunch,     1)
    assert     self.order_exists(self.client_, date.today() + one_day, self.lunch,     1)
    assert not self.order_exists(self.client_, date.today() + one_day, self.dinner,    1)

  def test_more_days(self):
    self.create_meals(number_of_days = 3)

    response = self.preorder_meals(client      = self.client_,
                                   canteen     = self.canteen,
                                   from_date   = date.today(),
                                   to_date     = date.today() + two_days,
                                   from_course = self.lunch.name,
                                   to_course   = self.lunch.name,
                                   token       = self.admin_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 2 + 3 + 2

    assert not self.order_exists(self.client_, date.today(),            self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today(),            self.lunch,     1)
    assert     self.order_exists(self.client_, date.today(),            self.dinner,    1)
    assert     self.order_exists(self.client_, date.today() + one_day,  self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today() + one_day,  self.lunch,     1)
    assert     self.order_exists(self.client_, date.today() + one_day,  self.dinner,    1)
    assert     self.order_exists(self.client_, date.today() + two_days, self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today() + two_days, self.lunch,     1)
    assert not self.order_exists(self.client_, date.today() + two_days, self.dinner,    1)

    assert not Order.objects.filter(meal__number__in = [2, 3]).exists()

  def test_only_courses(self):
    self.create_meals(number_of_days = 3)

    response = self.preorder_meals(client       = self.client_,
                                   canteen      = self.canteen,
                                   from_date    = date.today(),
                                   to_date      = date.today() + two_days,
                                   from_course  = self.breakfast.name,
                                   to_course    = self.dinner.name,
                                   only_courses = [self.lunch.name],
                                   token        = self.admin_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 3

    assert not self.order_exists(self.client_, date.today(),            self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today(),            self.lunch,     1)
    assert not self.order_exists(self.client_, date.today(),            self.dinner,    1)
    assert not self.order_exists(self.client_, date.today() + one_day,  self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today() + one_day,  self.lunch,     1)
    assert not self.order_exists(self.client_, date.today() + one_day,  self.dinner,    1)
    assert not self.order_exists(self.client_, date.today() + two_days, self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today() + two_days, self.lunch,     1)
    assert not self.order_exists(self.client_, date.today() + two_days, self.dinner,    1)

    assert not Order.objects.filter(meal__number__in = [2, 3]).exists()

  def test_except_courses(self):
    self.create_meals(number_of_days = 3)

    response = self.preorder_meals(client         = self.client_,
                                   canteen        = self.canteen,
                                   from_date      = date.today(),
                                   to_date        = date.today() + two_days,
                                   from_course    = self.breakfast.name,
                                   to_course      = self.dinner.name,
                                   except_courses = [self.lunch.name],
                                   token          = self.admin_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 6

    assert     self.order_exists(self.client_, date.today(),            self.breakfast, 1)
    assert not self.order_exists(self.client_, date.today(),            self.lunch,     1)
    assert     self.order_exists(self.client_, date.today(),            self.dinner,    1)
    assert     self.order_exists(self.client_, date.today() + one_day,  self.breakfast, 1)
    assert not self.order_exists(self.client_, date.today() + one_day,  self.lunch,     1)
    assert     self.order_exists(self.client_, date.today() + one_day,  self.dinner,    1)
    assert     self.order_exists(self.client_, date.today() + two_days, self.breakfast, 1)
    assert not self.order_exists(self.client_, date.today() + two_days, self.lunch,     1)
    assert     self.order_exists(self.client_, date.today() + two_days, self.dinner,    1)

    assert not Order.objects.filter(meal__number__in = [2, 3]).exists()

  def test_only_and_except_courses(self):
    self.create_meals(number_of_days = 3)

    response = self.preorder_meals(client         = self.client_,
                                   canteen        = self.canteen,
                                   from_date      = date.today(),
                                   to_date        = date.today() + two_days,
                                   from_course    = self.breakfast.name,
                                   to_course      = self.dinner.name,
                                   only_courses   = [self.lunch.name, self.dinner.name],
                                   except_courses = [self.dinner.name],
                                   token          = self.admin_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 3

    assert not self.order_exists(self.client_, date.today(),            self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today(),            self.lunch,     1)
    assert not self.order_exists(self.client_, date.today(),            self.dinner,    1)
    assert not self.order_exists(self.client_, date.today() + one_day,  self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today() + one_day,  self.lunch,     1)
    assert not self.order_exists(self.client_, date.today() + one_day,  self.dinner,    1)
    assert not self.order_exists(self.client_, date.today() + two_days, self.breakfast, 1)
    assert     self.order_exists(self.client_, date.today() + two_days, self.lunch,     1)
    assert not self.order_exists(self.client_, date.today() + two_days, self.dinner,    1)

    assert not Order.objects.filter(meal__number__in = [2, 3]).exists()

  def test_no_days(self):
    response = self.preorder_meals(client      = self.client_,
                                   canteen     = self.canteen,
                                   from_date   = date.today(),
                                   to_date     = date.today() - one_day,
                                   from_course = self.lunch.name,
                                   to_course   = self.lunch.name,
                                   token       = self.admin_token)

    assert response.status_code    == 400
    # TODO: translate this
    assert response.data['detail'] == 'Invalid date selection: from_date is later than to_date'

  def test_no_course(self):
    response = self.preorder_meals(client      = self.client_,
                                   canteen     = self.canteen,
                                   from_date   = date.today(),
                                   to_date     = date.today(),
                                   from_course = self.dinner.name,
                                   to_course   = self.lunch.name,
                                   token       = self.admin_token)

    assert response.status_code    == 400
    # TODO: translate this
    assert response.data['detail'] == 'Invalid course selection: from_course is later than to_course'

  def create_meals(self, number_of_days):
    counter = Counter()

    Meal.objects.bulk_create(Meal(name    = fake.sentence(3)[:30],
                                  date    = date.today() + timedelta(days = day),
                                  canteen = self.canteen,
                                  course  = course,
                                  number  = number + 1,
                                  stock_card_number = counter.fetch_and_add())
                             for day    in range(number_of_days)
                             for course in Course.objects.all()
                             for number in range(3))

  def order_exists(self, client, date, course, number):
    return Order.objects.filter(client = client,
                                meal__date   = date,
                                meal__course = course,
                                meal__number = number).exists()
