from datetime import date, timedelta
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_client_auth.tests.helpers import ClientAuthHelpers
from alto_django_kredit.models import Client
from alto_django_canteen.models import Canteen, CourseGroup, Course, Meal, Order
from alto_django_canteen.serializers import OrderSerializer
from alto_django_canteen.tests.helpers import CanteenHelpers
from alto_django_canteen.tests.factories import MealFactory, CourseFactory

User     = get_user_model()
today    = date.today()
tomorrow = today + timedelta(days = 1)

class BulkOrderingTests(APITestCase, ClientAuthHelpers, CanteenHelpers):
  def setUp(self):
    self.client_    = Client.objects.create(client_id = '123', card_id = '123',
                                            email = 'client@test.cz')
    self.user       = User.objects.create_user('test', 'test@test.cz', 'test',
                                               client = self.client_)
    self.user_token = self.get_token(username = 'test',  password = 'test' ).data['token']

    self.canteen1   = Canteen.objects.create(name = 'canteen1')
    self.canteen2   = Canteen.objects.create(name = 'canteen2')

    self.lunch      = CourseGroup.objects.get(name = 'lunch')

    self.breakfast  = Course.objects.get(name   = 'breakfast')
    self.lunch_main = CourseFactory.create(name = 'lunch_main', group = self.lunch)
    self.lunch_side = CourseFactory.create(name = 'lunch_side', group = self.lunch,
                                           is_additional = True)
    self.dinner     = Course.objects.get(name   = 'dinner')

    for canteen in [self.canteen1, self.canteen2]:
      for date in [today, tomorrow]:
        for course in [self.breakfast, self.lunch_side, self.dinner]:
          MealFactory.create_many(3, canteen = canteen, date = date, course = course)

        MealFactory.create_many(3, canteen = canteen, date = date, course = self.lunch_main,
                                additional_course = self.lunch_side)

  def test_creates_orders(self):
    orders = [
      {'canteen': self.canteen1.id,
       'date':    today.isoformat(),
       'group':   'breakfast',
       'orders':  [{'course': 'breakfast', 'number': 1}]},
      {'canteen': self.canteen1.id,
       'date':    tomorrow.isoformat(),
       'group':   'dinner',
       'orders':  [{'course': 'dinner', 'number': 2}]},
      {'canteen': self.canteen2.id,
       'date':    today.isoformat(),
       'group':   'lunch',
       'orders':  [
         {'course': 'lunch_main', 'number': 2},
         {'course': 'lunch_side', 'number': 3},
       ]}
    ]

    response = self.bulk_order(orders, self.user_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 4

    order1 = Order.objects.get(client             = self.client_,
                               meal__canteen      = self.canteen1,
                               meal__date         = today,
                               meal__course__name = 'breakfast')

    order2 = Order.objects.get(client             = self.client_,
                               meal__canteen      = self.canteen1,
                               meal__date         = tomorrow,
                               meal__course__name = 'dinner')

    order3 = Order.objects.get(client             = self.client_,
                               meal__canteen      = self.canteen2,
                               meal__date         = today,
                               meal__course__name = 'lunch_main')

    order4 = Order.objects.get(client             = self.client_,
                               meal__canteen      = self.canteen2,
                               meal__date         = today,
                               meal__course__name = 'lunch_side')

    assert order1.meal.number == 1
    assert order2.meal.number == 2
    assert order3.meal.number == 2
    assert order4.meal.number == 3

    assert response.data == { 'orders': OrderSerializer(Order.objects.all(), many = True).data }

  def test_updates_existing_orders(self):
    meal  = Meal.objects.get(canteen = self.canteen1, date = today,
                             course = 'breakfast', number = 1)
    order = Order.objects.create(client = self.client_, meal = meal)

    orders = [
      {'canteen': self.canteen1.id,
       'date':    today.isoformat(),
       'group':   'breakfast',
       'orders':  [{'course': 'breakfast', 'number': 2}]},
    ]

    response = self.bulk_order(orders, self.user_token)

    order.refresh_from_db()

    assert response.status_code  == 200
    assert Order.objects.count() == 1
    assert order.meal.number     == 2

  def test_cancels_orders_for_null_numbers(self):
    meal  = Meal.objects.get(canteen = self.canteen1, date = today,
                             course = 'breakfast', number = 1)
    order = Order.objects.create(client = self.client_, meal = meal)

    orders = [
      {'canteen': self.canteen1.id,
       'date':    today.isoformat(),
       'group':   'breakfast',
       'orders':  [{'course': 'breakfast', 'number': None}]},
    ]

    response = self.bulk_order(orders, self.user_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 0

  def test_nonexistent_meal_number(self):
    orders = [
      {'canteen': self.canteen1.id,
       'date':    today.isoformat(),
       'group':   'breakfast',
       'orders':  [{'course': 'breakfast', 'number': 5}]},
      {'canteen': self.canteen1.id,
       'date':    tomorrow.isoformat(),
       'group':   'dinner',
       'orders':  [{'course': 'dinner', 'number': 2}]},
    ]

    response = self.bulk_order(orders, self.user_token)

    assert response.status_code == 400
    assert response.data        == [
      {'orders': [{'course': 'breakfast',
                   'number': 'Jídlo s číslem 5 neexistuje'}]},
      {} # second order is OK
    ]

  def test_orders_exist_in_other_canteen(self):
    meal = Meal.objects.get(canteen = self.canteen2, date = today,
                            course = self.breakfast, number = 1)
    Order.objects.create(client = self.client_, meal = meal)

    orders = [
      {'canteen': self.canteen1.id,
       'date':    today.isoformat(),
       'group':   'breakfast',
       'orders':  [{'course': 'breakfast', 'number': 1}]},
      {'canteen': self.canteen1.id,
       'date':    tomorrow.isoformat(),
       'group':   'dinner',
       'orders':  [{'course': 'dinner', 'number': 2}]},
    ]

    response = self.bulk_order(orders, self.user_token)

    assert response.status_code == 400
    assert response.data        == [
      {'canteen': ['Existují objednávky v jiné jídelně'], 'code': 'orders_in_other_canteen_exist'},
      {} # second order is OK
    ]

  def test_unorder_one_canteen_and_order_in_another(self):
    meal = Meal.objects.get(canteen = self.canteen2, date = today,
                            course = self.breakfast, number = 1)
    Order.objects.create(client = self.client_, meal = meal)

    orders = [
      {'canteen': self.canteen1.id,
       'date':    today.isoformat(),
       'group':   'breakfast',
       'orders':  [{'course': 'breakfast', 'number': 1}]},
      {'canteen': self.canteen2.id,
       'date':    today.isoformat(),
       'group':   'breakfast',
       'orders':  [{'course': 'breakfast', 'number': None}]},
    ]

    response = self.bulk_order(orders, self.user_token)

    assert response.status_code == 200

  def test_additional_course_not_ordered_no_default(self):
    orders = [
      {'canteen': self.canteen1.id,
       'date':    today.isoformat(),
       'group':   'breakfast',
       'orders':  [{'course': 'breakfast', 'number': 1}]},
      {'canteen': self.canteen2.id,
       'date':    today.isoformat(),
       'group':   'lunch',
       'orders':  [{'course': 'lunch_main', 'number': 2}]}
    ]

    response = self.bulk_order(orders, self.user_token)

    assert response.status_code == 400
    assert response.data        == [
      {}, # first order is OK
      {'orders': [{'course': 'lunch_main', 'code': 'additional_meal_not_ordered',
                   'detail': 'Doplňkové jídlo nebylo objednáno'}]}
    ]
    assert Order.objects.count() == 0 # first order was rolled back

  def test_additional_course_ordered_but_no_main(self):
    orders = [
      {'canteen': self.canteen1.id,
       'date':    today.isoformat(),
       'group':   'breakfast',
       'orders':  [{'course': 'breakfast', 'number': 1}]},
      {'canteen': self.canteen2.id,
       'date':    today.isoformat(),
       'group':   'lunch',
       'orders':  [{'course': 'lunch_side', 'number': 2}]}
    ]

    response = self.bulk_order(orders, self.user_token)

    assert response.status_code == 400
    assert response.data        == [
      {}, # first order is OK
      {'orders': [{'course': 'lunch_side', 'code': 'main_meal_not_ordered',
                   'detail': 'Hlavní jídlo nebylo objednáno'}]}
    ]
