import datetime, factory
from pydash import py_
from faker  import Faker
from alto_django_utils.counter  import Counter
from alto_django_canteen.models import Canteen, Meal, Course, CourseGroup

fake    = Faker()
counter = Counter()

class classproperty:
  def __init__(self, fget):
    self.fget = fget

  def __get__(self, owner_self, owner_cls):
    return self.fget(owner_cls)

class CardNumberFactory(factory.DictFactory):
  card_number = factory.Sequence(lambda n: str(n))

class CanteenFactory:
  @classproperty
  def default(cls):
    return Canteen.objects.get_or_create(name = 'default canteen')[0]

  @classproperty
  def other(cls):
    return Canteen.objects.get_or_create(name = 'other canteen')[0]

class CourseGroupFactory:
  @classmethod
  def create(cls, *, name, display_name = None, served_at = None,
             lock_before = None, order_number = None):
    display_name = display_name or name
    served_at    = datetime.time(hour = 12)
    lock_before  = datetime.timedelta()
    last_group   = CourseGroup.objects.order_by('order_number').last()
    order_number = last_group.order_number + 1 if last_group else 1

    return CourseGroup.objects.create(name         = name,
                                      display_name = display_name,
                                      served_at    = served_at,
                                      lock_before  = lock_before,
                                      order_number = order_number)

  @classproperty
  def breakfast(cls):
    return CourseGroup.objects.get_or_create(name = 'breakfast')[0]

  @classproperty
  def lunch(cls):
    return CourseGroup.objects.get_or_create(name = 'lunch')[0]

  @classproperty
  def dinner(cls):
    return CourseGroup.objects.get_or_create(name = 'dinner')[0]

class CourseFactory:
  @classmethod
  def create(cls, *, name, group, display_name = None, order_number = None, is_additional = False):
    display_name = display_name or name
    last_course  = Course.objects.order_by('order_number').last()
    order_number = last_course.order_number + 1 if last_course else 1

    return Course.objects.create(name          = name,
                                 group         = group,
                                 display_name  = display_name,
                                 order_number  = order_number,
                                 is_additional = is_additional)

  @classproperty
  def breakfast(cls):
    return Course.objects.get_or_create(name = 'breakfast',
                                        group = CourseGroupFactory.breakfast,
                                        defaults = {'display_name': 'breakfast',
                                                    'order_number': 1})[0]

  @classproperty
  def lunch_soup(cls):
    return Course.objects.get_or_create(name = 'lunch_soup',
                                        group = CourseGroupFactory.lunch,
                                        defaults = {'display_name': 'soup',
                                                    'order_number': 4})[0]

  @classproperty
  def lunch_main(cls):
    return Course.objects.get_or_create(name = 'lunch_main',
                                        group = CourseGroupFactory.lunch,
                                        defaults = {'display_name': 'main',
                                                    'order_number': 5})[0]

  @classproperty
  def lunch_side(cls):
    return Course.objects.get_or_create(name = 'lunch_side',
                                        group = CourseGroupFactory.lunch,
                                        defaults = {'display_name': 'side',
                                                    'order_number': 6})[0]

  @classproperty
  def dinner(cls):
    return Course.objects.get_or_create(name = 'dinner',
                                        group = CourseGroupFactory.dinner,
                                        defaults = {'display_name': 'dinner',
                                                    'order_number': 9})[0]

class MealFactory:
  @classmethod
  def build(cls, *, name = None, date = None, canteen = None,
             course = None, number = None, stock_card_number = None,
             additional_course = None, additional_course_default_number = None):
    canteen = canteen or Canteen.objects.first()
    course  = course  or Course.objects.first()
    date    = date    or datetime.date.today()
    name    = name    or fake.sentence(3)[:64]

    if number == None:
      last_meal = Meal.objects \
        .filter(canteen = canteen, course = course, date = date) \
        .order_by('number').last()

      number = last_meal.number + 1 if last_meal else 1

    return Meal(name              = name,
                date              = date,
                canteen           = canteen,
                course            = course,
                number            = number,
                stock_card_number = counter.fetch_and_add(),
                additional_course = additional_course,
                additional_course_default_number = additional_course_default_number)

  @classmethod
  def create(cls, **kwargs):
    meal = cls.build(**kwargs)
    meal.save()
    return meal

  @classmethod
  def build_many(cls, count, **kwargs):
    return [cls.build(**py_.assign(kwargs, { 'number': i + 1 })) for i in range(count)]

  @classmethod
  def create_many(cls, count, **kwargs):
    return Meal.objects.bulk_create(cls.build_many(count, **kwargs))
