import pydash
from django.utils.translation import gettext_lazy as _
from rest_framework.serializers import (Serializer, IntegerField, DateField,
                                        PrimaryKeyRelatedField, ValidationError)
from dynamic_rest.fields import DynamicRelationField
from alto_django_utils.validators import IsInValidator
from alto_django_utils.serializers import serializer_for
from alto_django_canteen.models import Canteen, CourseGroup, Course, Meal, Order
from .models import Terminal

OrderSerializer       = serializer_for(Order)
CourseSerializer      = serializer_for(Course)
CanteenSerializer     = serializer_for(Canteen)
TerminalSerializer    = serializer_for(Terminal)
CourseGroupSerializer = serializer_for(CourseGroup)
BaseMealSerializer    = serializer_for(Meal)

additional_course_validator = \
  IsInValidator(Course.objects.filter(is_additional = True).all,
                message = _('course %(value)s is not an additional course'))

class MealSerializer(BaseMealSerializer):
  additional_course = DynamicRelationField(CourseSerializer,
                                           validators = [additional_course_validator])

class OrderCourseSerializer(Serializer):
  course = PrimaryKeyRelatedField(queryset = Course.objects.all())
  number = IntegerField(allow_null = True)

class OrderCourseGroupSerializer(Serializer):
  date    = DateField()
  canteen = PrimaryKeyRelatedField(queryset = Canteen.objects.all())
  group   = PrimaryKeyRelatedField(queryset = CourseGroup.objects.all())
  orders  = OrderCourseSerializer(many = True)

  def validate(self, data):
    duplicates = pydash.duplicates([order['course'] for order in data['orders']])

    if duplicates:
      message = _("Duplicate orders for courses %(courses)s") % \
        {'courses': [course.name for course in duplicates] }

      raise ValidationError({ 'orders': message })

    data['orders'] = [{'course': order['course'],
                       'meal':   self._get_meal(data['date'], data['canteen'],
                                                order['course'], order['number'])}
                      for order in data['orders']]

    return data

  def _get_meal(self, date, canteen, course, number):
    if number:
      try:
        return Meal.objects.get(canteen = canteen,
                                date    = date,
                                course  = course,
                                number  = number)
      except Meal.DoesNotExist as error:
        message = _("Meal with_number %(number)s does not exist") % { 'number': number }

        raise ValidationError({ 'orders': [{ 'course': course.name, 'number': message }] })

class CancelCourseGroupSerializer(Serializer):
  date    = DateField()
  canteen = PrimaryKeyRelatedField(queryset = Canteen.objects.all())
  group   = PrimaryKeyRelatedField(queryset = CourseGroup.objects.all())
