from datetime import time, timedelta
from django.conf import settings
from django.db.models import (
  Model, ForeignKey, CharField, TextField, BooleanField, DateField, TimeField, DateTimeField,
  DurationField, SmallIntegerField, PositiveIntegerField, PositiveSmallIntegerField,
  GenericIPAddressField, SET_NULL)
from alto_django_utils.models import PROTECT
from alto_django_kredit.models import Client

class Canteen(Model):
  class Meta:
    db_table            = 'canteens'
    verbose_name        = 'jídelna'
    verbose_name_plural = 'jídelny'

  name                 = CharField(max_length = 32)
  register_number      = SmallIntegerField(null = True, blank = True)
  discriminator_html   = TextField(blank = True)
  discriminator_height = PositiveIntegerField(null = True, blank = True)

  def __repr__(self):
    return f"<{self.__class__.__name__} '{self.name}'>"

  def __str__(self):
    return self.name

class Terminal(Model):
  class Meta:
    db_table = 'terminals'

  ip              = GenericIPAddressField(unique = True)
  name            = CharField(max_length = 32, unique = True)
  secret          = CharField(max_length = 36, default = '')
  registered_at   = DateTimeField(auto_now_add = True)
  default_canteen = ForeignKey(Canteen, related_name = 'terminals', on_delete = SET_NULL,
                               null = True, blank = True)

  def __repr__(self):
    return f"<{self.__class__.__name__} {self.name} {self.default_canteen}>"

  __str__ = __repr__

class CourseGroup(Model):
  class Meta:
    db_table            = 'course_groups'
    verbose_name        = 'skupina chodů'
    verbose_name_plural = 'skupiny chodů'

  name         = CharField(max_length = 16, primary_key = True)
  display_name = CharField(max_length = 64)
  served_at    = TimeField()
  lock_before  = DurationField(default = timedelta)
  order_number = PositiveSmallIntegerField(unique = True)

  def __repr__(self):
    return f"<{self.__class__.__name__} {self.name}>"

  def __str__(self):
    return self.name

class Course(Model):
  class Meta:
    db_table            = 'courses'
    verbose_name        = 'chod'
    verbose_name_plural = 'chody'

  name          = CharField(max_length = 16, primary_key = True)
  group         = ForeignKey(CourseGroup, related_name = 'courses', on_delete = PROTECT,
                             verbose_name = 'skupina')
  display_name  = CharField(max_length = 64)
  order_number  = PositiveSmallIntegerField(unique = True)
  is_additional = BooleanField(default = False)

  def __repr__(self):
    return f"<{self.__class__.__name__} {self.name}>"

  def __str__(self):
    return self.name

class Meal(Model):
  class Meta:
    db_table            = 'meals'
    verbose_name        = 'jídlo'
    verbose_name_plural = 'jídla'
    unique_together     = ('canteen', 'date', 'course', 'number')

  name              = CharField(max_length = 250)
  description       = CharField(max_length = 250, default = '', blank = True)
  canteen           = ForeignKey(Canteen, related_name = 'meals', on_delete = PROTECT,
                                 verbose_name = 'jídelna')
  course            = ForeignKey(Course,  related_name = 'meals', on_delete = PROTECT,
                                 verbose_name = 'chod')
  date              = DateField()
  number            = PositiveSmallIntegerField()
  show_on_web       = BooleanField(default = True)
  stock_card_number = PositiveIntegerField()
  additional_course = ForeignKey(Course, related_name = '+', on_delete = PROTECT, null = True,
                                 verbose_name = 'dodatečný chod')
  additional_course_default_number = PositiveSmallIntegerField(null = True)

  def __repr__(self):
    return (f"<{self.__class__.__name__} {self.date} '{self.canteen.name}' "
            f"{self.course.name} {self.number} '{self.name}'>")

  def __str__(self):
    return self.name

class Order(Model):
  class Meta:
    db_table            = 'orders'
    verbose_name        = 'objednávka'
    verbose_name_plural = 'objednávky'
    unique_together     = ('client', 'meal')

  client     = ForeignKey(Client, related_name = 'orders', on_delete = PROTECT,
                          verbose_name = 'klient')
  meal       = ForeignKey(Meal,   related_name = 'orders', on_delete = PROTECT,
                          verbose_name = 'jídlo')
  ordered    = PositiveSmallIntegerField(default = 1)
  eaten      = PositiveSmallIntegerField(default = 0)
  created_at = DateTimeField(auto_now_add = True)
  updated_at = DateTimeField(auto_now = True)

  def __repr__(self):
    return f"<{self.__class__.__name__} {repr(self.client)} {repr(self.meal)} {self.ordered} {self.eaten}>"

  __str__ = __repr__

class CanteenPermissions(Model):
  class Meta:
    managed = False

    permissions = [
       ('order_meals', 'Order meals')
    ]
