from decimal import Decimal
from datetime import timedelta
from django.urls import reverse
from django.utils import timezone
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_kredit.models import Client, Transaction
from alto_django_client_auth.tests.helpers import ClientAuthHelpers, auth_header

User = get_user_model()

class TransactionsTests(APITestCase, ClientAuthHelpers):
  def setUp(self):
    admin        = User.objects.create_user('admin', 'admin@test.cz', 'admin', is_superuser = True)
    self.token   = self.get_token(username = 'admin', password = 'admin').data['token']
    self.client_ = Client.objects.create(client_id = '123', card_id = '123',
                                         email = 'client@test.cz')

  def test_create_with_neither_amount_nor_balance(self):
    response = self.client.post(reverse('alto_django_kredit:transactions-list'),
                                { 'client': self.client_.id,
                                  'kind': 'KC' },
                                **auth_header(self.token))

    message = "Alespoň jedna z hodnot 'amount' a 'balance' musí být zadána."

    assert response.status_code == 400
    assert message in response.data['amount']
    assert message in response.data['balance']

  def test_create_with_only_amount_first_transaction(self):
    response = self.client.post(reverse('alto_django_kredit:transactions-list'),
                                { 'client': self.client_.id,
                                  'kind': 'KC',
                                  'amount': '1.23' },
                                **auth_header(self.token))

    transaction = Transaction.objects.last()

    assert response.status_code == 201
    assert transaction.amount   == Decimal('1.23')
    assert transaction.balance  == Decimal('1.23')

  def test_create_with_only_amount_second_transaction(self):
    Transaction.objects.create(client = self.client_, kind = 'KC',
                               amount = '1.23', balance = '1.23')

    response = self.client.post(reverse('alto_django_kredit:transactions-list'),
                                { 'client': self.client_.id,
                                  'kind': 'KC',
                                  'amount': '3.21' },
                                **auth_header(self.token))

    transaction = Transaction.objects.last()

    assert response.status_code == 201
    assert transaction.amount   == Decimal('3.21')
    assert transaction.balance  == Decimal('4.44')

  def test_create_with_only_balance_first_transaction(self):
    response = self.client.post(reverse('alto_django_kredit:transactions-list'),
                                { 'client': self.client_.id,
                                  'kind': 'KC',
                                  'balance': '1.23' },
                                **auth_header(self.token))

    transaction = Transaction.objects.last()

    assert response.status_code == 201
    assert transaction.amount   == Decimal('1.23')
    assert transaction.balance  == Decimal('1.23')

  def test_create_with_only_balance_second_transaction(self):
    Transaction.objects.create(client = self.client_, kind = 'KC',
                               amount = '1.23', balance = '1.23')

    response = self.client.post(reverse('alto_django_kredit:transactions-list'),
                                { 'client': self.client_.id,
                                  'kind': 'KC',
                                  'balance': '4.44' },
                                **auth_header(self.token))

    transaction = Transaction.objects.last()

    assert response.status_code == 201
    assert transaction.amount   == Decimal('3.21')
    assert transaction.balance  == Decimal('4.44')

  def test_create_with_matching_amount_and_balance(self):
    response = self.client.post(reverse('alto_django_kredit:transactions-list'),
                                { 'client': self.client_.id,
                                  'kind': 'KC',
                                  'amount': '1.23',
                                  'balance': '1.23' },
                                **auth_header(self.token))

    transaction = Transaction.objects.last()

    assert response.status_code == 201
    assert transaction.amount   == Decimal('1.23')
    assert transaction.balance  == Decimal('1.23')

  def test_create_with_non_matching_amount_and_balance(self):
    response = self.client.post(reverse('alto_django_kredit:transactions-list'),
                                { 'client': self.client_.id,
                                  'kind': 'KC',
                                  'amount': '1.23',
                                  'balance': '2.34' },
                                **auth_header(self.token))

    assert response.status_code == 400
    assert "Neodpovídá 'balance' po odečtení aktuálního zůstatku." in response.data['amount']
    assert "Neodpovídá aktuálnímu zůstatku po přičtení 'amount'."  in response.data['balance']

  def test_create_with_transacted_at_before_last_transaction(self):
    Transaction.objects.create(client = self.client_, kind = 'KC',
                               amount = 100, balance = 100) # transacted_at = now

    response = self.client.post(reverse('alto_django_kredit:transactions-list'),
                                { 'client': self.client_.id,
                                  'kind': 'KC',
                                  'amount': 100,
                                  'balance': 200,
                                  'transacted_at': timezone.now() - timedelta(hours = 1)},
                                **auth_header(self.token))

    assert response.status_code == 400
    assert 'Existuje novější transakce' in response.data['transacted_at'][0]

  def test_update_is_disallowed(self):
    transaction = Transaction.objects.create(client = self.client_, kind = 'KC',
                                             amount = 100, balance = 100)

    response = self.client.put(reverse('alto_django_kredit:transactions-detail',
                                        args = [transaction.id]),
                                {},
                                **auth_header(self.token))

    assert response.status_code    == 405
    assert response.data['detail'] == 'Changing transactions is not allowed.'

  def test_partial_update_is_disallowed(self):
    transaction = Transaction.objects.create(client = self.client_, kind = 'KC',
                                             amount = 100, balance = 100)

    response = self.client.patch(reverse('alto_django_kredit:transactions-detail',
                                        args = [transaction.id]),
                                {},
                                **auth_header(self.token))

    assert response.status_code    == 405
    assert response.data['detail'] == 'Changing transactions is not allowed.'

  def test_delete_is_disallowed(self):
    transaction = Transaction.objects.create(client = self.client_, kind = 'KC',
                                             amount = 100, balance = 100)

    response = self.client.delete(reverse('alto_django_kredit:transactions-detail',
                                        args = [transaction.id]),
                                **auth_header(self.token))

    assert response.status_code    == 405
    assert response.data['detail'] == 'Deleting transactions is not allowed.'
