import pytest
from contextlib import contextmanager
from django.test import LiveServerTestCase
from alto_django_kredit.models import Client, Transaction
from alto_django_kredit.settings import settings
from alto_django_kredit.tests.helpers import AmbiguousClientFinder
from alto_django_kredit.tests.pages.show_balance import ShowBalancePage

# NOTES:
# LiveServerTestCase doesn't restore initial db data after test, only at the end of test suite run
# -> tests that rely on preexisting Courses will fail if run after these

@pytest.mark.slow
@pytest.mark.live
@pytest.mark.nodbreset
class ShowBalanceTests(LiveServerTestCase):
  def page(self):
    return ShowBalancePage(self.live_server_url)

  def setUp(self):
    super().setUp()
    settings.client_finder = AmbiguousClientFinder()

  def tearDown(self):
    settings.reset()
    super().tearDown()

  def test_success(self):
    client = Client.objects.create(client_id = '123456', card_id = '123456', email = 'test@test.cz')
    Transaction.objects.create(client = client, kind = 'POI', amount = 100, balance = 100)

    with self.page() as page:
      page.enter_and_submit('123456', 'test@test.cz')

      assert page.get_balance() == 100

  def test_invalid_card_number(self):
    with self.page() as page:
      page.enter_and_submit('1234567', 'test@test.cz')

      assert page.errors_for('card_number') == 'Není platné číslo karty'

  def test_invalid_email(self):
    with self.page() as page:
      page.enter_and_submit('123456', 'test@test')

      assert page.errors_for('email') == 'Vložte platnou e-mailovou adresu.'

  def test_client_not_found(self):
    with self.page() as page:
      page.enter_and_submit('123456', 'test@test.cz')

      assert page.non_field_errors() == 'Klient se zadanými údaji nebyl nalezen.'

  def test_multiple_clients_found(self):
    Client.objects.create(client_id = '123456', card_id = '123456', email = 'test@test.cz')
    Client.objects.create(client_id = '223456', card_id = '223456', email = 'test@test.cz')

    with self.page() as page:
      page.enter_and_submit('23456', 'test@test.cz')

      assert 'Bylo nalezeno několik klientů se zadanými údaji.' in page.non_field_errors()
