from datetime import timedelta
from django.urls import reverse
from django.utils import timezone
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_kredit.models import Client, Transaction
from alto_django_client_auth.tests.helpers import ClientAuthHelpers, auth_header

User = get_user_model()

class GetBalancesTests(APITestCase, ClientAuthHelpers):
  def setUp(self):
    User.objects.create_user('admin', 'admin@test.cz', 'admin', is_superuser = True)
    self.token = self.get_token(username = 'admin', password = 'admin').data['token']

  def create_client_and_transactions(self):
    client = Client.objects.create(client_id = '123', card_id = '123', email = 'client@test.cz')

    Transaction.objects.create(client = client, kind = 'KC',  amount = '1.23', balance = '1.23',
                               transacted_at = timezone.now())
    Transaction.objects.create(client = client, kind = 'POI', amount = '2.34', balance = '2.34',
                               transacted_at = timezone.now())
    Transaction.objects.create(client = client, kind = 'POI', amount = '3.45', balance = '3.45',
                               transacted_at = timezone.now() - timedelta(days = 1))

    return client

  def test_without_at_param(self):
    client   = self.create_client_and_transactions()
    path     = reverse('alto_django_kredit:get-balances')
    response = self.client.get(path + f'?client_id={client.id}', **auth_header(self.token))

    self.assertEqual(response.status_code, 200)
    self.assertEqual(response.data['balances'], {'KC': '1.23', 'POI': '2.34'})

  def test_with_at_param(self):
    client   = self.create_client_and_transactions()
    path     = reverse('alto_django_kredit:get-balances')
    at       = timezone.now() - timedelta(hours = 12)
    response = self.client.get(
      path, { 'client_id': client.id, 'at': at.isoformat() }, **auth_header(self.token)
    )

    self.assertEqual(response.status_code, 200)
    self.assertEqual(response.data['balances'], {'KC': '0', 'POI': '3.45'})

  def test_with_nonexistent_client(self):
    path     = reverse('alto_django_kredit:get-balances')
    response = self.client.get(path + '?client_id=123', **auth_header(self.token))

    self.assertEqual(response.status_code, 404)
