from django.db.models.functions import Lower
from alto_django_kredit.models import Client, Transaction

class ClientFinder:
  class InvalidCardNumber(Exception): pass
  class ClientNotFound(Exception): pass
  class MultipleClientsFound(Exception): pass
  class BillNotFound(Exception): pass
  class MultipleBillsFound(Exception): pass
  class BillDoesNotBelongToClient(Exception): pass

  def clients_for_card(self, card_number):
    if self.is_valid_card_number(card_number):
      return Client.objects.filter(card_id__iexact = card_number)
    else:
      raise self.InvalidCardNumber()

  def client_for_card_and_bill(self, card_number, bill_number, bill_date):
    clients = self.clients_for_card(card_number)

    if not clients:
      raise self.ClientNotFound()

    try:
      transaction = self._find_transaction(bill_number, bill_date)
    except Transaction.DoesNotExist:
      raise self.BillNotFound()
    except Transaction.MultipleObjectsReturned:
      raise self.MultipleBillsFound()

    try:
      return clients.get(transactions = transaction)
    except Client.DoesNotExist:
      raise self.BillDoesNotBelongToClient()

  def client_for_card_and_email(self, card_number, email):
    clients = self.clients_for_card(card_number).filter(email = email)

    try:
      return clients.get()
    except Client.DoesNotExist:
      raise self.ClientNotFound()
    except Client.MultipleObjectsReturned:
      raise self.MultipleClientsFound()

  def find_unused_card(self):
    card = Client.objects.unassigned.order_by('pk').first()

    if not card: raise self.ClientNotFound()

    return card

  def is_valid_card_number(self, card_number):
    return True

  def _find_transaction(self, bill_number, bill_date):
    return Transaction.objects.get(
      kind                = 'POI',
      description__regex  = f'0*{bill_number}',
      transacted_at__date = bill_date
    )
