from rest_framework.generics import ListAPIView
from rest_framework.exceptions import NotAuthenticated, AuthenticationFailed
from rest_framework.permissions import AllowAny
from dynamic_rest.viewsets import WithDynamicViewSetMixin
from alto_django_utils.params import required_body_param
from alto_django_utils.pagination import BigCustomPagination
from alto_django_utils.exceptions import Misconfigured
from alto_django_settings.models import Setting
from alto_django_kredit.models import Client
from alto_django_kredit.settings import settings
from alto_django_kredit.serializers import ClientSerializer
from alto_django_base_auth.views.login import GetAuthToken, GetTokenPair
from alto_django_client_auth.serializers import ClientSerializerForSelect
from alto_django_client_auth.services.profile import EnsureProfileComplete

# login by username and password
class GetClientTokenMixin:
  def _get_context(self, request):
    context = super()._get_context(request)
    context = self._get_client(context)

    return context

  def _get_result(self, context):
    result = super()._get_result(context)

    result['client'] = ClientSerializer(context['client']).data if context['client'] else None
    if 'profile_complete' in context: result['profile_complete'] = context['profile_complete']

    return result

  def _get_client(self, context):
    context = context.copy()
    client  = context['user'].client

    if client:
      context['client'] = client

      try:
        EnsureProfileComplete(client)()

        context['profile_complete'] = True
      except (EnsureProfileComplete.MissingValue, EnsureProfileComplete.MissingField) as error:
        context['profile_complete'] = False
      except EnsureProfileComplete.InvalidRequiredFieldsSetting as error:
        raise Misconfigured(str(error))

    else:
      context['client'] = None

    return context

class GetClientToken    (GetClientTokenMixin, GetAuthToken): pass
class GetClientTokenPair(GetClientTokenMixin, GetTokenPair): pass

# login by card number and password
class GetClientTokenByCardMixin:
  def _find_user(self, request):
    try:
      card_number = required_body_param(request, 'card_number')
      clients     = settings.client_finder.clients_for_card(card_number)
      client      = clients.get(user__isnull = False)

      return client.user
    except Client.DoesNotExist:
      raise AuthenticationFailed({ 'detail': 'bad credentials' })

class GetClientTokenByCard    (GetClientTokenByCardMixin, GetClientToken    ): pass
class GetClientTokenPairByCard(GetClientTokenByCardMixin, GetClientTokenPair): pass

class GetClientsForSelect(WithDynamicViewSetMixin, ListAPIView):
  ordering           = ('surname', 'firstname', 'client_id')
  queryset           = Client.objects.filter(user__isnull = False, user__is_active = True)
  serializer_class   = ClientSerializerForSelect
  pagination_class   = BigCustomPagination
  permission_classes = [AllowAny]

  def list(self, request, *args, **kwargs):
    setting = Setting.for_key('auth.login.client_select', {})

    if (request.user and request.user.is_superuser) or not setting.get('must_authenticate', True):
      return super().list(request, *args, **kwargs)
    else:
      raise NotAuthenticated

get_client_token              = GetClientToken.as_view()
get_client_token_pair         = GetClientTokenPair.as_view()
get_client_token_by_card      = GetClientTokenByCard.as_view()
get_client_token_pair_by_card = GetClientTokenPairByCard.as_view()
