import pytest, re
from django.urls                         import reverse
from pydash                              import py_
from alto_django_base_auth.tests.helpers import auth_header
from alto_django_settings.models         import Setting

@pytest.fixture
def incomplete_card_user_credentials(card_credentials_factory):
  return card_credentials_factory()

@pytest.fixture
def incomplete_card_user(incomplete_card_user_credentials, client_factory, user_factory):
  client = client_factory(card_id = incomplete_card_user_credentials['card_number'],
                          firstname = '', surname = '', email = '')

  return user_factory(password = incomplete_card_user_credentials['password'],
                      client = client, username = 'incomplete_card_user',
                      first_name = 'Incomplete Card', last_name = 'User',
                      email = 'incomplete_card_user@test.cz')

pytestmark = pytest.mark.django_db()

@pytest.mark.parametrize(
  '  url_name,                       use_jwt, token_key',
  (('get-client-token-by-card',      False,   'token'),
   ('v1-get-client-token-by-card',   False,   'token'),
   ('get-client-token-pair-by-card', True,    'access_token')))
def test_client_authentication_by_card(
  api_client, url_name, use_jwt, token_key, card_user, card_user_credentials
):
  get_response = api_client.post(
    reverse(f'alto_django_client_auth:{url_name}'), card_user_credentials)

  assert get_response.status_code == 200

  test_response = api_client.get(
    reverse('alto_django_base_auth:test-token'),
    **auth_header(token = get_response.data[token_key], use_jwt = use_jwt))

  assert test_response.status_code == 200

def test_refresh_token(auth_helpers, api_client, card_user, card_user_credentials):
  get_response = api_client.post(
    reverse('alto_django_client_auth:get-client-token-pair-by-card'), card_user_credentials)

  refresh_response = auth_helpers.refresh_token(get_response.data['refresh_token'])

  assert refresh_response.status_code == 200

@pytest.mark.parametrize(
  'url_name',
  ('get-client-token-by-card', 'v1-get-client-token-by-card', 'get-client-token-pair-by-card'))
class TestCommonClientAuthCases:
  def test_card_client_presence(self, api_client, url_name, card_user, card_user_credentials):
    response = api_client.post(
      reverse(f'alto_django_client_auth:{url_name}'), card_user_credentials)

    assert response.status_code               == 200
    assert response.data['user']['username']  == card_user.username
    assert response.data['client']['card_id'] == card_user.client.card_id

  def test_with_complete_profile(self, api_client, url_name, card_user, card_user_credentials):
    response = api_client.post(
      reverse(f'alto_django_client_auth:{url_name}'), card_user_credentials)

    assert response.status_code              == 200
    assert response.data['profile_complete'] == True

  def test_with_incomplete_profile(self, api_client, url_name,
                                   incomplete_card_user, incomplete_card_user_credentials):
    Setting.objects.create(key   = 'auth.profile.required_fields',
                           value = ['firstname', 'surname', 'email'])

    response = api_client.post(
      reverse(f'alto_django_client_auth:{url_name}'), incomplete_card_user_credentials)

    assert response.status_code              == 200
    assert response.data['profile_complete'] == False

  def test_nonexistent_card_number(self, api_client, url_name, card_user_credentials):
    response = api_client.post(
      reverse(f'alto_django_client_auth:{url_name}'), card_user_credentials)

    assert response.status_code    == 403
    assert response.data['detail'] == 'bad credentials'

  def test_bad_password(self, api_client, url_name, card_user, invalid_card_user_credentials):
    response = api_client.post(
      reverse(f'alto_django_client_auth:{url_name}'), invalid_card_user_credentials)

    assert response.status_code    == 403
    assert response.data['detail'] == 'bad credentials'

  def test_unregistered_client(self, api_client, url_name, client, card_user_credentials):
    response = api_client.post(
      reverse(f'alto_django_client_auth:{url_name}'), card_user_credentials)

    assert response.status_code    == 403
    assert response.data['detail'] == 'bad credentials'

  def test_inactive_user(self, api_client, url_name, inactive_card_user,
                         inactive_card_user_credentials):
    response = api_client.post(
      reverse(f'alto_django_client_auth:{url_name}'), inactive_card_user_credentials)

    assert response.status_code    == 403
    assert response.data['detail'] == 'user not activated'
