import pytest
from django.urls                         import reverse
from alto_django_base_auth.tests.helpers import auth_header
from alto_django_settings.models         import Setting

@pytest.fixture
def incomplete_client_user_credentials(username_credentials_factory):
  return username_credentials_factory(username = 'incomplete_client_user')

@pytest.fixture
def incomplete_client_user(incomplete_client_user_credentials, client_factory, user_factory):
  client = client_factory(firstname = '', surname = '', email = '')

  return user_factory(**incomplete_client_user_credentials, client = client,
                      first_name = 'Incomplete Client', last_name = 'User',
                      email = 'incomplete_client_user@test.cz')

pytestmark = pytest.mark.django_db()

@pytest.mark.parametrize(
  '  url_name,               use_jwt, token_key',
  (('get-client-token',      False,   'token'),
   ('v1-get-client-token',   False,   'token'),
   ('get-client-token-pair', True,    'access_token')))
def test_client_authentication(api_client, url_name, use_jwt, token_key,
                               client_user, client_user_credentials):
  get_response = api_client.post(
    reverse(f'alto_django_client_auth:{url_name}'), client_user_credentials)

  assert get_response.status_code == 200

  test_response = api_client.get(
    reverse('alto_django_base_auth:test-token'),
    **auth_header(token = get_response.data[token_key], use_jwt = use_jwt))

  assert test_response.status_code == 200

def test_refresh_token(auth_helpers, api_client, client_user, client_user_credentials):
  get_response = api_client.post(
    reverse('alto_django_client_auth:get-client-token-pair'), client_user_credentials)

  refresh_response = auth_helpers.refresh_token(get_response.data['refresh_token'])

  assert refresh_response.status_code == 200

@pytest.mark.parametrize(
  'url_name',
  ('get-client-token', 'v1-get-client-token', 'get-client-token-pair'))
class TestCommonClientAuthCases:
  def test_client_user_presence(self, api_client, url_name, client_user, client_user_credentials):
    response = api_client.post(reverse(f'alto_django_client_auth:{url_name}'),
                               client_user_credentials)

    assert response.status_code               == 200
    assert response.data['user']['username']  == client_user.username
    assert response.data['client']['card_id'] == client_user.client.card_id

  def test_with_complete_profile(self, api_client, url_name, client_user,
                                 client_user_credentials):
    response = api_client.post(reverse(f'alto_django_client_auth:{url_name}'),
                               client_user_credentials)

    assert response.status_code              == 200
    assert response.data['profile_complete'] is True

  def test_with_incomplete_profile(self, api_client, url_name, incomplete_client_user,
                                   incomplete_client_user_credentials):
    Setting.objects.create(key   = 'auth.profile.required_fields',
                           value = ['firstname', 'surname', 'email'])

    response = api_client.post(reverse(f'alto_django_client_auth:{url_name}'),
                               incomplete_client_user_credentials)

    assert response.status_code              == 200
    assert response.data['profile_complete'] is False

  def test_with_admin_user(self, api_client, url_name, admin_user, admin_user_credentials):
    response = api_client.post(reverse(f'alto_django_client_auth:{url_name}'),
                               admin_user_credentials)

    assert response.status_code                  == 200
    assert response.data['user']['is_superuser'] is True
    assert response.data['client']               is None
