import re
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_kredit.models import Client
from alto_django_kredit.settings import settings
from alto_django_kredit.tests.factories.client import ClientFactory
from alto_django_kredit.tests.helpers import AmbiguousClientFinder
from alto_django_client_auth.tests.helpers import ClientAuthHelpers

User = get_user_model()

class ClientByCardTests(APITestCase, ClientAuthHelpers):
  def setUp(self):
    settings.client_finder = AmbiguousClientFinder()

  def tearDown(self):
    settings.reset()

  def test_valid_email(self):
    client = ClientFactory.create(card_id = '123', email = 'test@test.cz')

    response = self.client_by_card('123')

    assert response.data['found']      == 'one'
    assert response.data['registered'] == False
    assert response.data['email']      == 't**t@test.cz'

  def test_no_email(self):
    client = ClientFactory.create(card_id = '123', email = None)

    response = self.client_by_card('123')

    assert response.data['found']      == 'one'
    assert response.data['registered'] == False
    assert response.data['email']      == None

  def test_invalid_email(self):
    client = ClientFactory.create(card_id = '123', email = 'abcd')

    response = self.client_by_card('123')

    assert response.data['found']      == 'one'
    assert response.data['registered'] == False
    assert response.data['email']      == None

  def test_registered(self):
    client = ClientFactory.create(card_id = '123', email = None)
    user   = User.objects.create_user('test', 'test@test.cz', 'test', client = client)

    response = self.client_by_card('123')

    assert response.data['found']      == 'one'
    assert response.data['registered'] == True

  def test_not_found(self):
    response = self.client_by_card('123')

    assert response.data['found'] == 'none'

  def test_ambiguous_card_number(self):
    client1 = ClientFactory.create(client_id = '112345', card_id = '112345', email = None)
    client2 = ClientFactory.create(client_id = '212345', card_id = '212345', email = None)

    response = self.client_by_card('12345')

    assert response.data['found']          == 'many'
    assert response.data['all_registered'] == False

  def test_ambiguous_card_number_all_registered(self):
    client1 = ClientFactory.create(client_id = '112345', card_id = '112345', email = None)
    client2 = ClientFactory.create(client_id = '212345', card_id = '212345', email = None)
    user1   = User.objects.create_user('test1', 'test1@test.cz', 'test1', client = client1)
    user2   = User.objects.create_user('test2', 'test2@test.cz', 'test2', client = client2)

    response = self.client_by_card('12345')

    assert response.data['found']          == 'many'
    assert response.data['all_registered'] == True

  def test_invalid_card_number(self):
    response = self.client_by_card('1234567')

    assert response.status_code         == 400
    assert response.data['card_number'] == ['Není platné číslo karty']
